IF NOT EXISTS (SELECT * FROM sys.xml_schema_collections WHERE xml_schema_collections.name = 'extended_events_xml_schema_collection')
BEGIN
	CREATE XML SCHEMA COLLECTION dbo.extended_events_xml_schema_collection
	AS
	N'<xs:schema attributeFormDefault="unqualified" elementFormDefault="qualified" xmlns:xs="http://www.w3.org/2001/XMLSchema">
		<xs:element name="event">
		<xs:complexType>
			<xs:sequence>
			<xs:element name="data" maxOccurs="unbounded" minOccurs="0">
				<xs:complexType>
				<xs:sequence>
					<xs:element type="xs:string" name="value"/>
					<xs:element type="xs:string" name="text" minOccurs="0"/>
				</xs:sequence>
				<xs:attribute type="xs:string" name="name" use="optional"/>
				</xs:complexType>
			</xs:element>
			<xs:element name="action" maxOccurs="unbounded" minOccurs="0">
				<xs:complexType>
				<xs:sequence>
					<xs:element type="xs:string" name="value"/>
				</xs:sequence>
				<xs:attribute type="xs:string" name="name" use="optional"/>
				<xs:attribute type="xs:string" name="package" use="optional"/>
				</xs:complexType>
			</xs:element>
			</xs:sequence>
			<xs:attribute type="xs:string" name="name"/>
			<xs:attribute type="xs:string" name="package"/>
			<xs:attribute type="xs:dateTime" name="timestamp"/>
		</xs:complexType>
		</xs:element>
	</xs:schema>';
END

IF NOT EXISTS (SELECT * FROM sys.tables WHERE tables.name = 'extended_events_xml')
BEGIN
	CREATE TABLE dbo.extended_events_xml
	(	extended_events_xml_id INT NOT NULL IDENTITY(1,1) CONSTRAINT PK_extended_events_xml PRIMARY KEY CLUSTERED,
		sample_time_utc DATETIME2(3) NOT NULL,
		event_data_xml XML(dbo.extended_events_xml_schema_collection) NOT NULL)
	WITH (DATA_COMPRESSION = PAGE);

	CREATE NONCLUSTERED INDEX NCI_extended_events_xml_sample_time_utc ON dbo.extended_events_xml (sample_time_utc);

	CREATE PRIMARY XML INDEX PXMLI_extended_events_xml_event_data_xml ON dbo.extended_events_xml (event_data_xml);
END

IF NOT EXISTS (SELECT * FROM sys.tables WHERE tables.name = 'extended_events_data')
BEGIN
	CREATE TABLE dbo.extended_events_data
	(	extended_events_data_id INT NOT NULL IDENTITY(1,1) CONSTRAINT PK_extended_events_data PRIMARY KEY CLUSTERED,
		sample_time_utc DATETIME2(3) NOT NULL,
		database_name VARCHAR(128) NOT NULL,
		event_name VARCHAR(50) NOT NULL,
		session_id SMALLINT NOT NULL,
		cpu_time BIGINT NOT NULL,
		duration BIGINT NOT NULL,
		physical_reads BIGINT NOT NULL,
		logical_reads BIGINT NOT NULL,
		writes BIGINT NOT NULL,
		row_count BIGINT NOT NULL,
		client_app_name VARCHAR(128) NOT NULL,
		client_host_name VARCHAR(128) NOT NULL,
		username VARCHAR(128) NOT NULL);

	CREATE NONCLUSTERED INDEX NCI_extended_events_data_sample_time_utc ON dbo.extended_events_data (sample_time_utc)
	WITH (DATA_COMPRESSION = PAGE);
END

IF EXISTS (SELECT * FROM sys.procedures WHERE procedures.name = 'process_extended_events')
BEGIN
	DROP PROCEDURE dbo.process_extended_events;
END
GO

CREATE PROCEDURE dbo.process_extended_events
	@extended_events_session_name VARCHAR(MAX),
	@extended_events_file_path VARCHAR(MAX),
	@stop_and_drop_all_extended_events_sessions_and_files BIT
AS
BEGIN
	SET NOCOUNT ON;
	-- Add a trailing forward slash to the path name, if needed.
	IF RIGHT(@extended_events_file_path, 1) <> '\'
	BEGIN
		SELECT @extended_events_file_path = @extended_events_file_path + '\';
	END
	-- Variables to hold info about the Extended Events session
	DECLARE @current_extended_events_session_name VARCHAR(MAX);
	DECLARE @does_event_session_exist BIT = 0;
	DECLARE @is_event_session_started BIT = 0;

	IF EXISTS (SELECT * FROM sys.dm_xe_sessions WHERE dm_xe_sessions.name LIKE @extended_events_session_name + '%')
	BEGIN
		SELECT @does_event_session_exist = 1;
		SELECT @is_event_session_started = 1;
		SELECT
			@current_extended_events_session_name = dm_xe_sessions.name
		FROM sys.dm_xe_sessions WHERE dm_xe_sessions.name LIKE @extended_events_session_name + '%';
	END
	ELSE
	IF EXISTS (SELECT * FROM sys.server_event_sessions WHERE server_event_sessions.name LIKE @extended_events_session_name + '%')
	BEGIN
		SELECT @does_event_session_exist = 1;
		SELECT
			@current_extended_events_session_name = server_event_sessions.name
		FROM sys.server_event_sessions WHERE server_event_sessions.name LIKE @extended_events_session_name + '%';
	END

	DECLARE @session_number_previous INT;
	DECLARE @session_number_next INT;

	IF @does_event_session_exist = 1
	BEGIN
		SELECT @session_number_previous =
			CASE
				WHEN ISNUMERIC(RIGHT(@current_extended_events_session_name, 1)) = 1
				AND LEN(@current_extended_events_session_name) <> LEN(@extended_events_session_name)
					THEN SUBSTRING(@current_extended_events_session_name, LEN(@extended_events_session_name) + 1, LEN(@current_extended_events_session_name) - LEN('query_metrics'))
				ELSE 0
			END
	END
	SELECT @session_number_next = @session_number_previous + 1;
	
	DECLARE @next_extended_events_session_name VARCHAR(MAX)  = @extended_events_session_name + CAST(@session_number_next AS VARCHAR(MAX));

	-- Create Extended Events commands for use later on.
	DECLARE @extended_events_session_create_command NVARCHAR(MAX);
	SELECT @extended_events_session_create_command = '
		CREATE EVENT SESSION ' + @next_extended_events_session_name + ' ON SERVER
		ADD EVENT sqlserver.rpc_completed (
			ACTION (
				sqlserver.client_app_name,
				sqlserver.client_hostname,
				sqlserver.database_name,
				sqlserver.session_id,
				sqlserver.username)
			WHERE (sqlserver.client_app_name NOT LIKE ''SQLAgent%'')),
		ADD EVENT sqlserver.sql_batch_completed (
			ACTION (
				sqlserver.client_app_name,
				sqlserver.client_hostname,
				sqlserver.database_name,
				sqlserver.session_id,
				sqlserver.username)
			WHERE (sqlserver.client_app_name NOT LIKE ''SQLAgent%''))
			ADD TARGET package0.event_file
				(SET FILENAME = ''' + @extended_events_file_path + @next_extended_events_session_name + '.xel'',
					MAX_FILE_SIZE = 1000, -- 1000MB
					MAX_ROLLOVER_FILES = 3)
			WITH (	EVENT_RETENTION_MODE = ALLOW_SINGLE_EVENT_LOSS,
					MAX_DISPATCH_LATENCY = 15 SECONDS,
					MAX_MEMORY = 1024MB,
					STARTUP_STATE = OFF);';
	-- Command to start the new Extended Events session
	DECLARE @extended_events_session_start_command NVARCHAR(MAX);
	SELECT @extended_events_session_start_command = '
		ALTER EVENT SESSION ' + @next_extended_events_session_name + ' ON SERVER
		STATE = START;';
	-- Command to stop the previous Extended Events session
	DECLARE @extended_events_session_stop_command NVARCHAR(MAX);
	SELECT @extended_events_session_stop_command = '
		ALTER EVENT SESSION ' + @current_extended_events_session_name + ' ON SERVER
		STATE = STOP;';
	-- Command to delete *.xel files from the previous event session
	DECLARE @extended_events_session_file_delete_command NVARCHAR(MAX);
	SELECT @extended_events_session_file_delete_command = 'EXEC xp_cmdshell ''DEL ' + @extended_events_file_path + @current_extended_events_session_name + '*.xel /q'',no_output;';
	-- Command to delete *.xel files from any previous event sessions
	DECLARE @extended_events_session_file_delete_all_command NVARCHAR(MAX);
	SELECT @extended_events_session_file_delete_all_command = 'EXEC xp_cmdshell ''DEL ' + @extended_events_file_path + @extended_events_session_name + '*.xel /q'',no_output;';
	-- Command to drop an Extended Events session
	DECLARE @extended_events_session_drop_command NVARCHAR(MAX);
	SELECT @extended_events_session_drop_command = '
		DROP EVENT SESSION ' + @current_extended_events_session_name + ' ON SERVER;';
	-- Command to read Extended Events XML into a table
	DECLARE @extended_events_read_data_command NVARCHAR(MAX);
	SELECT
		@extended_events_read_data_command = '
		INSERT INTO dbo.extended_events_xml
			(sample_time_utc, event_data_xml)
		SELECT
			timestamp_utc AS sample_time_utc,
			CAST(event_data AS VARCHAR(MAX)) AS event_data_xml
		FROM sys.fn_xe_file_target_read_file(''' + @extended_events_file_path + @current_extended_events_session_name + '*.xel'', NULL, NULL, NULL);';

	-- With these commands created, execute Extended Events logic accordingly:
	IF @stop_and_drop_all_extended_events_sessions_and_files = 1
	BEGIN -- Stop and drop Extended Events session and delete files, regardless of status
		IF @is_event_session_started = 1
		BEGIN
			EXEC sp_executesql @extended_events_session_stop_command;
		END
		IF @does_event_session_exist = 1
		BEGIN
			EXEC sp_executesql @extended_events_session_drop_command;
		END

		EXEC sp_executesql @extended_events_session_file_delete_all_command;
	END
	ELSE
	IF @does_event_session_exist = 0
	BEGIN -- If Extended Events session doesn't exist, then delete any files, and then create/start the new session
		EXEC sp_executesql @extended_events_session_file_delete_all_command;
		
		EXEC sp_executesql @extended_events_session_create_command;
		EXEC sp_executesql @extended_events_session_start_command;
	END
	ELSE
	BEGIN
		IF @is_event_session_started = 1 -- If the previous session is started, then stop it
		BEGIN
			EXEC sp_executesql @extended_events_session_stop_command;
		END
		-- Create a new session and start it
		EXEC sp_executesql @extended_events_session_create_command;
		EXEC sp_executesql @extended_events_session_start_command;
		-- Now that the new session is collecting events, process the data from the previous session
		EXEC sp_executesql @extended_events_read_data_command;
		
		EXEC sp_executesql @extended_events_session_drop_command;
		EXEC sp_executesql @extended_events_session_file_delete_command;
	END
	-- Crunch the XML into query details
	IF EXISTS (SELECT * FROM dbo.extended_events_xml)
	BEGIN
		INSERT INTO dbo.extended_events_data
			(sample_time_utc, database_name, event_name, session_id, cpu_time, duration, physical_reads,
			 logical_reads, writes, row_count, client_app_name, client_host_name, username)
		SELECT
			sample_time_utc,
			event_data_xml.value('(event/action[@name="database_name"]/value)[1]', 'SYSNAME') AS database_name,
			event_data_xml.value('(event/@name)[1]', 'VARCHAR(50)') As event_name,
			event_data_xml.value('(event/action[@name="session_id"]/value)[1]', 'SMALLINT') AS session_id,
			event_data_xml.value('(event/data[@name="cpu_time"]/value)[1]', 'BIGINT') AS cpu_time,
			event_data_xml.value('(event/data[@name="duration"]/value)[1]', 'BIGINT') AS duration,
			event_data_xml.value('(event/data[@name="physical_reads"]/value)[1]', 'BIGINT') AS physical_reads,
			event_data_xml.value('(event/data[@name="logical_reads"]/value)[1]', 'BIGINT') AS logical_reads,
			event_data_xml.value('(event/data[@name="writes"]/value)[1]', 'BIGINT') AS writes,
			event_data_xml.value('(event/data[@name="row_count"]/value)[1]', 'BIGINT') AS row_count,
			event_data_xml.value('(event/action[@name="client_app_name"]/value)[1]', 'VARCHAR(128)') AS client_app_name,
			event_data_xml.value('(event/action[@name="client_hostname"]/value)[1]', 'VARCHAR(128)') AS client_host_name,
			event_data_xml.value('(event/action[@name="username"]/value)[1]', 'SYSNAME') AS username
		FROM dbo.extended_events_xml;
		
		TRUNCATE TABLE dbo.extended_events_xml;
	END
END
GO

EXEC dbo.process_extended_events
	@extended_events_session_name = 'Sql_Server_Query_Metrics',
	@extended_events_file_path = 'C:\SQLBackup\',
	@stop_and_drop_all_extended_events_sessions_and_files = 1;
GO

/*	QA & Troubleshooting Queries

SELECT GETUTCDATE() AS the_time, COUNT(*), MIN(sample_time_utc) AS sample_time_utc_min , MAX(sample_time_utc) AS sample_time_utc_max
FROM dbo.extended_events_data WITH (NOLOCK);

SELECT COUNT(*) FROM dbo.extended_events_data;
SELECT COUNT(*) FROM dbo.extended_events_xml;

SELECT * FROM dbo.extended_events_xml
SELECT * FROM dbo.extended_events_data

TRUNCATE TABLE dbo.extended_events_data;
TRUNCATE TABLE dbo.extended_events_xml;

DROP TABLE dbo.extended_events_data;
DROP TABLE dbo.extended_events_xml;

*/

